﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Spectral1.DATA_ACCESS
{
    public class DA_3DWireframe
    {
        #region "========= DECLARATIONS =================================="
        public const int waveform_plot_width = 800;
        public const int harmonic_plot_width = 800;
        public const int harmonic_y_offset = 50;
        public const int cycles = 5;
        public const float waveform_overall_scaling = 0.5f;
        public const float harmonic_overall_scaling = 0.5f;
        public const int num_interpolates = 5;
        public const int waveform_viewport_distance = 1000;
        public const int harmonic_viewport_distance = 1500;
        public const double waveform_viewport_zoom = 0.95;
        public const double harmonic_viewport_zoom = 0.9;

        #endregion

        #region "======= SUB-CLASSES 3D ==================================="
        public class Point3D
        {
            public Single X { get; set; }
            public Single Y { get; set; }
            public Single Z { get; set; }

            public bool IsNull { get; set; }

            public Point3D(Single _X, Single _Y, Single _Z)
            {
                X = _X;
                Y = _Y;
                Z = _Z;
                IsNull = false;
            }
        }

        public class PolyLine3D
        {
            public List<Point3D> PointList { get; set; }
            public Pen LinePen { get; set; }
            public Pen HighlightedLinePen { get; set; }

            public PolyLine3D()
            {
                PointList = new List<Point3D>();
            }
        }

        public class Rectangle3D
        {
            public Point3D P0 { get; set; }
            public Point3D P1 { get; set; }
            public Point3D P2 { get; set; }
            public Point3D P3 { get; set; }

            public Rectangle3D()
            {
                P0 = new Point3D(0,0,0);
                P1 = new Point3D(0, 0, 0);
                P2 = new Point3D(0, 0, 0);
                P3 = new Point3D(0, 0, 0);
            }
        }

        public class World3D
        {
            public List<PolyLine3D> PolyLinesList { get; set; }
            public World3D()
            {
                PolyLinesList = new List<PolyLine3D>();
            }
        }

        #endregion

        #region "======= SUB-CLASSES 2D ==================================="
        public class PolyLine2D
        {
            public List<Point> PointList { get; set; }
            public Pen LinePen { get; set; }
            public Pen HighlightedLinePen { get; set; }
            public PolyLine2D()
            {
                PointList = new List<Point>();
            }
        }
        public class World2D
        {
            public List<PolyLine2D> PolyLinesList { get; set; }
            public Rectangle ViewPort { get; set; }
            public int ViewPortDistance { get; set; }

            public Single rotateX{ get; set; }
            public Single rotateY{ get; set; }
            public Single rotateZ{ get; set; }
            public Double ZoomPerc{ get; set; }

            public World2D()
            {
                PolyLinesList = new List<PolyLine2D>();
                ViewPort = new Rectangle();
                rotateX = 0;
                rotateY = 0;
                rotateZ = 0;
                ZoomPerc = 1;
            }
        }
        #endregion

        #region "======= DECLARATIONS =============================="
        public World3D w3d = new World3D();
        public World2D w2d = new World2D();
        #endregion

        #region "======= METHODS ==================================="
        public DA_3DWireframe()
        {
        }

        private PolyLine2D ConvertPolyLineto2D(PolyLine3D p3d)
        {
            PolyLine2D pl2D = new PolyLine2D();
            for (Int32 i = 0; i < p3d.PointList.Count(); i++)
            {
                Point p = new Point();
                p.X = (int)p3d.PointList[i].X;
                p.Y = (int)p3d.PointList[i].Y;
                pl2D.PointList.Add(p);
            }
            pl2D.LinePen = p3d.LinePen;
            pl2D.HighlightedLinePen = p3d.HighlightedLinePen;

            return pl2D;
        }

        public void ClearWorld()
        {
            w3d.PolyLinesList.Clear();
            w2d.PolyLinesList.Clear();
        }

        private bool IsInside(Point3D p)
        {
            if (p.IsNull == true) { return false; }
            return ((p.X <= w2d.ViewPort.Right) && (p.X >= w2d.ViewPort.Left) && (p.Y <= w2d.ViewPort.Bottom) && (p.Y >= w2d.ViewPort.Top));
        }

        private Point3D LineIntersection(Point3D l1s, Point3D l1e, Point3D l2s, Point3D l2e)
        {
            Point3D cp = new Point3D(0,0,0);

            Single m1 = 0;
            Single c1 = 0;
            Single m2 = 0;
            Single c2 = 0;
            Single deltax = 0;
            Single deltay = 0;

            deltax = l1e.X - l1s.X;
            deltay = l1e.Y - l1s.Y;
            if (deltax == 0) { deltax = (Single)0.0001; }

            m1 = deltay / deltax;
            c1 = l1e.Y - (m1 * l1e.X);


            deltax = l2e.X - l2s.X;
            deltay = l2e.Y - l2s.Y;
            if (deltax == 0) { deltax = (Single)0.001; }

            m2 = deltay / deltax;
            c2 = l2e.Y - (m2 * l2e.X);


            if ((m1 - m2) == 0)
            {
                cp.X = l1s.X;
                cp.Y = l2s.Y;
                return cp;
            }

            cp.X = (Single)((c2-c1) / (m1-m2));
            cp.Y = (Single)((m1 * cp.X) + c1);

            return cp;
        }

        private Point3D GetClippingPoint(Point3D points3d, Point3D pointe3d)
        {
            Point3D ps3d;
            Point3D pe3d;

            Point3D cp1 = new Point3D(0,0,0);
            Point3D cp2 = new Point3D(0,0,0);

            Point3D vptr = new Point3D(w2d.ViewPort.Right, w2d.ViewPort.Top,0);
            Point3D vpbr= new Point3D(w2d.ViewPort.Right, w2d.ViewPort.Bottom, 0);
            Point3D vpbl = new Point3D(w2d.ViewPort.Left, w2d.ViewPort.Bottom, 0);
            Point3D vptl = new Point3D(w2d.ViewPort.Left, w2d.ViewPort.Top, 0);

            if (IsInside(points3d))
            {
                ps3d = points3d;
                pe3d = pointe3d;
            }
            else
            {
                pe3d = points3d;
                ps3d = pointe3d;
            }


            
            if (pe3d.X > w2d.ViewPort.Right)
            {
                if (pe3d.Y < w2d.ViewPort.Top)
                {
                    //Right Top
                    cp1 = LineIntersection(ps3d,pe3d, vptr, vpbr);
                    cp2 = LineIntersection(ps3d, pe3d, vptr, vptl);
                    if (IsInside(cp1) == true){ return cp1; }else { return cp2; }
                }
                else if (pe3d.Y > w2d.ViewPort.Bottom)
                {
                    //Right Bottom
                    cp1 = LineIntersection(ps3d, pe3d, vptr, vpbr);
                    cp2 = LineIntersection(ps3d, pe3d, vpbr, vpbl);
                    if (IsInside(cp1) == true) { return cp1; } else { return cp2; }

                }
                else
                {
                    //Right Mid
                    return LineIntersection(ps3d, pe3d, vptr, vpbr);
                }
            }
            else if (pe3d.X < w2d.ViewPort.Left)
            {
                if (pe3d.Y < w2d.ViewPort.Top)
                {
                    //Left Top
                    cp1 = LineIntersection(ps3d, pe3d, vptl, vpbl);
                    cp2 = LineIntersection(ps3d, pe3d, vptr, vptl);
                    if (IsInside(cp1) == true) { return cp1; } else { return cp2; }
                }
                else if (pe3d.Y > w2d.ViewPort.Bottom)
                {
                    //Left Bottom
                    cp1 = LineIntersection(ps3d, pe3d, vptl, vpbl);
                    cp2 = LineIntersection(ps3d, pe3d, vpbr, vpbl);
                    if (IsInside(cp1) == true) { return cp1; } else { return cp2; }
                }
                else
                {
                    //Left Mid
                    return LineIntersection(ps3d, pe3d, vptl, vpbl);
                }
            }
            else
            {
                if (pe3d.Y < w2d.ViewPort.Top)
                {
                    //Mid Top
                    return LineIntersection(ps3d, pe3d, vptl, vptr);
                }
                else if (pe3d.Y > w2d.ViewPort.Bottom)
                {
                    //Mid Bottom
                    return LineIntersection(ps3d, pe3d, vpbl, vpbr);
                }
            }

            cp1.IsNull = true;
            return cp1; //Should never get here
        }

        private Point3D TransformPoint(Point3D p,Single rotateX, Single rotateY, Single rotateZ, Double ZoomPerc)
        {
            double x = p.X * ZoomPerc;
            double y = p.Y * ZoomPerc;
            double z = p.Z * ZoomPerc;
            double h;
            double rot; 

            //Rotate about X
            h = Math.Sqrt(z * z + y * y);
            rot = Math.Atan2(y, z);
            rot+= rotateX * Math.PI/180;
            z = (h * Math.Cos(rot));
            y = (h * Math.Sin(rot));

            //Rotate about Y
            h = Math.Sqrt(x * x + z * z);
            rot = Math.Atan2(z, x);
            rot+= rotateY * Math.PI / 180;
            x = (h * Math.Cos(rot));
            z = (h * Math.Sin(rot));

            //Rotate about Z
            h = Math.Sqrt(x * x + y * y);
            rot = Math.Atan2(y,x);
            rot += rotateZ * Math.PI / 180;
            x = (h * Math.Cos(rot));
            y = (h * Math.Sin(rot));

            Point3D tp = new Point3D((Single)x, (Single)y, (Single)z);
            return tp;
        }

        public static PolyLine3D get_interpolated_polyline(PolyLine3D p1, PolyLine3D p2, int interpolate_index, int num_interpolates, Pen interpolated_waveform_pen, Pen highlighted_interpolated_waveform_pen)
        {
            PolyLine3D wp_interpolated = new PolyLine3D();
            wp_interpolated.HighlightedLinePen = highlighted_interpolated_waveform_pen;
            wp_interpolated.LinePen = interpolated_waveform_pen;

            for (int i = 0; i < p1.PointList.Count; i++)
            {
                float z1 = p1.PointList[i].Z;
                float z2 = p2.PointList[i].Z;
                float zi = z1 + (((z2 - z1) / (num_interpolates + 1)) * (interpolate_index + 1));
                float y1 = p1.PointList[i].Y;
                float y2 = p2.PointList[i].Y;
                float yi = y1 + (((y2 - y1) / (num_interpolates + 1)) * (interpolate_index + 1));
                Point3D p = new Point3D(p1.PointList[i].X, yi, zi);
                wp_interpolated.PointList.Add(p);
            }

            return wp_interpolated;
        }


        public void RefreshWorld2D()
        {
            w2d.PolyLinesList.Clear();

            for (Int32 i = 0; i < w3d.PolyLinesList.Count; i++)
            {
                PolyLine3D PL3d = w3d.PolyLinesList[i];
                PolyLine3D PL3dT = new PolyLine3D();
                PL3dT.LinePen = PL3d.LinePen;
                PL3dT.HighlightedLinePen = PL3d.HighlightedLinePen;

                Point3D ps3d = TransformPoint(PL3d.PointList[0], w2d.rotateX, w2d.rotateY, w2d.rotateZ, w2d.ZoomPerc);
                Point3D pe3d;
                for (Int32 p = 1; p < PL3d.PointList.Count; p++)
                {
                    pe3d = TransformPoint(PL3d.PointList[p], w2d.rotateX, w2d.rotateY, w2d.rotateZ, w2d.ZoomPerc);

                    if ((IsInside(ps3d) == true))
                    {
                        PL3dT.PointList.Add(ps3d);
                        if (IsInside(pe3d) == false)
                        {
                            PL3dT.PointList.Add(GetClippingPoint(ps3d, pe3d));
                        }
                        else
                        {
                            PL3dT.PointList.Add(pe3d);
                        }
                    }
                    else
                    {
                        if ((IsInside(pe3d) == true))
                        {
                            PL3dT.PointList.Add(GetClippingPoint(ps3d, pe3d));
                            PL3dT.PointList.Add(pe3d);
                        }
                    }
                    ps3d = pe3d;
                }
                w2d.PolyLinesList.Add(ConvertPolyLineto2D(PL3dT));
            }
        }

        #endregion

        #region "========================== MACRO METHODS =============================="
        public void Add3DAxis(List<PolyLine3D> L,Pen axis_pen)
        {
            PolyLine3D wp3d = new PolyLine3D();
            wp3d.LinePen = axis_pen;
            wp3d.HighlightedLinePen = axis_pen;
            Point3D p = new Point3D(-400, 0, -200);
            wp3d.PointList.Add(p);
            p = new Point3D(-400, -100, -200);
            wp3d.PointList.Add(p);
            L.Add(wp3d);
        }

        public GraphicsPath GetStringPath(string s, float dpi, RectangleF rect, Font font, StringFormat format)
        {
            GraphicsPath path = new GraphicsPath();
            // Convert font size into appropriate coordinates
            float emSize = dpi * font.SizeInPoints / 72;
            path.AddString(s, font.FontFamily, (int)font.Style, emSize, rect, format);

            return path;
        }

        public void AddTextTo3DWorld(String s, List<PolyLine3D> L, Point3D location, RectangleF rect, float dpi, Pen text_pen)
        {
            const int char_spacing = 20;
            for (int i = 0; i < s.Length; i++)
            {
                Point3D char_location = new Point3D(location.X + (char_spacing * i), location.Y, location.Z);
                Char c = Convert.ToChar(s.Substring(i, 1));
                if (c.ToString().Trim().Length > 0)
                {
                    L.Add(GetCharAsPolyline(c, char_location,rect,dpi,text_pen));
                }
            }
        }

        private PolyLine3D GetCharAsPolyline(Char c, Point3D location, RectangleF rect,float dpi,Pen text_pen)
        {
            PolyLine3D text3d = new PolyLine3D();

            if (c.ToString().Trim().Length > 0)
            {
                GraphicsPath path = new GraphicsPath();
                string s = c.ToString();
              
                Font font = new Font("Courier New", 5, FontStyle.Regular, GraphicsUnit.Pixel);
                StringFormat format = StringFormat.GenericDefault;// StringFormat.GenericTypographic;
                path = GetStringPath(s, dpi, rect, font, format);

                for (int i = 0; i < path.PathPoints.GetLength(0); i++)
                {
                    PointF p2d = path.PathPoints[i];
                    Point3D p3d = new Point3D(p2d.X + location.X, p2d.Y + location.Y, location.Z);
                    text3d.PointList.Add(p3d);
                }
                text3d.LinePen = text_pen;
                text3d.HighlightedLinePen = text_pen;
            }

            return text3d;
        }
        #endregion

    }
}
